/**
 * \file: mspin_appl_if_adapter.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * MySpin ApplInterfaceAdapter
 *
 * \component: MSPIN
 *
 * \author: Torsten Plate ICT-ADITG/SW2 tplate@de.adit-jv.com
 *
 * \copyright: (c) 2003 - 2013 ADIT Corporation
 *
 * \history
 * 0.1 TPlate Initial version
 *
 ***********************************************************************/

#include "mspin_typedef.h"
#include "mspin_appl_if_adapter.h"
#include "mspin_connection_adapter.h"
#include "mspin_connection_adapter_iap2.h"
#include "mspin_lm_adapter.h"
#include "mspin_core_adapter.h"
#include "mspin_core_callback_adapter.h"
#include "mspin_logging.h"
#include "mspin_measurement.h"
#ifdef USE_GENERATED_MYSPIN_VERSION_FILE
#include "mspin_version.h"
#endif
#include <semaphore.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>  // sleep()
#include <errno.h>   // errno
#include <pthread.h>

//Enable define per compile flag when you want to enable IVI Core engineering functions
//#define MYSPIN_IVI_CORE_ENGINEERING //can be enabled also as compile flag...


#define MAXBUFLEN 255

#define VEHICLE_DATA_KEY_GEOLOCATION_NMEA 1
#define VEHICLE_DATA_KEY_IS_MOVING 2
#define VEHICLE_DATA_KEY_IS_NIGHT 3

//ToDo: Is the following required??? To be checked!
/* PRQA: Lint Message 160, 505, 578: deactivation because DLT macros generate this lint findings*/
/*lint -e160 -e505 -e578*/
static U32 gInstanceCounter = 0;

MSPIN_ERROR MSPIN_Init(void)
{
    MSPIN_ERROR retConn = MSPIN_SUCCESS;
    MSPIN_ERROR ret = MSPIN_SUCCESS;
    mspin_log_registerWithDlt();

    // Workaround: since ilm_init()/Galcore might reset DLT app, we would loose earlier trace messages
    // so first call mspin_lm_init() and afterwards log with DLT

    ret = mspin_lm_init();

#ifdef MYSPIN_GIT_VERSION
    mspin_log_printLn(eMspinVerbosityInfo,
            "%s() mySPIN TA version %s", __FUNCTION__, MYSPIN_GIT_VERSION); //print version info
#else
    mspin_log_printLn(eMspinVerbosityInfo,
            "%s() mySPIN TA version %s", __FUNCTION__, "n/a"); //Git version not available
#endif //#ifdef MYSPIN_GIT_VERSION

#if (MSPIN_IVI_CORE_VERSION < 0x01020000UL) //Renamed to mySpin_GetVersion since IVI Core 1.2
    mspin_log_printLn(eMspinVerbosityInfo, "%s() Core version %s", __FUNCTION__, mySpin_getVersion());
#else
    mspin_log_printLn(eMspinVerbosityInfo, "%s() Core version %s", __FUNCTION__, mySpin_GetVersion());
#if (MSPIN_IVI_CORE_VERSION > 0x01020100UL) //Since IVI Core 1.2.2 OS and CPU info are supported
    mspin_log_printLn(eMspinVerbosityInfo, "%s() Core OS and CPU info are '%s'", __FUNCTION__, mySpin_GetSystemIdentification());
#endif //
#endif //(MSPIN_IVI_CORE_VERSION < 0x01020000UL)

    retConn = mspin_conn_init();
    if (ret == MSPIN_SUCCESS)   // failure of mspin_lm_init() is more important - don't overwrite
    {
        ret = retConn;
    }

    mySpin_Init();
    return ret;
}

void MSPIN_Shutdown(void)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s()", __FUNCTION__);
    mySpin_Shutdown();
    mspin_lm_shutdown();

    mspin_log_unregisterWithDlt();
}

MSPIN_Instance_t* MSPIN_CreateInstance(void* context)
{
    mspin_layerManager_context_t *pLMContext = malloc(sizeof(mspin_layerManager_context_t));
    if (pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(callingCtx=%p){%d} Creating LayerManager context=%p",
                __FUNCTION__, context, gInstanceCounter + 1, pLMContext);

        memset(pLMContext, 0, sizeof(mspin_layerManager_context_t));

        //Only set default values which aren't identical to '0'
        pLMContext->useWaylandTouch = TRUE;

        int rc = pthread_rwlock_init(&(pLMContext->rwlock), NULL);
        if (rc)
        {
            mspin_log_printLn(eMspinVerbosityFatal,
                    "%s() FATAL ERROR: Failed to init rwlock with rc=%d",
                    __FUNCTION__, rc);
            free(pLMContext);
            pLMContext = NULL;
            return NULL;
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(callingCtx=%p){%d} FATAL ERROR: Failed to allocate memory for LayerManager Context",
                __FUNCTION__, context, gInstanceCounter + 1);

        return NULL;
    }

    mspin_context_t *pContext = malloc(sizeof(mspin_context_t));
    if (pContext)
    {
        memset(pContext, 0, sizeof(mspin_context_t));

        pContext->instanceId = ++gInstanceCounter;

        mspin_log_printLn(eMspinVerbosityDebug, "%s(callingCtx=%p){%d} context=%p created",
                __FUNCTION__, context, pContext->instanceId, pContext);

        //Only set values which aren't identical to '0'
        pContext->outsideContext = context;
        pContext->cancelReason = MSPIN_ERROR_UNKNOWN;
        pContext->initializationTimeout = 10;
        pContext->pLayerManagerContext = pLMContext;

        int rc = pthread_mutex_init(&(pContext->LMCtxMutex), NULL);
        if (0 != rc)
        {
            mspin_log_printLn(eMspinVerbosityFatal,
                    "%s() FATAL ERROR: Failed to init LMCtxMutex with rc=%d",
                    __FUNCTION__, rc);
            free(pLMContext);
            pLMContext = NULL;
            free(pContext);
            pContext = NULL;
            return NULL;
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(callingCtx=%p){%d} FATAL ERROR: Failed to allocate memory",
                __FUNCTION__, context, gInstanceCounter + 1);
    }

    return (MSPIN_Instance_t*)pContext;
}

void MSPIN_DeleteInstance(MSPIN_Instance_t **ppInstanceHandle)
{
    mspin_context_t *pContext = (mspin_context_t*)(*ppInstanceHandle);

    if (pContext)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(context=%p){%d}",
               __FUNCTION__, *ppInstanceHandle, pContext->instanceId);
        //Reset session values to allow restart of session
        pContext->EAPSessionStarted = FALSE;
        pContext->isReady = FALSE;
        pContext->isCanceled = FALSE;
        pContext->cancelReason = MSPIN_ERROR_UNKNOWN;
        pContext->hasEverBeenReady = FALSE;

        void* pCore = mspin_core_stopCoreInstance(pContext);

        if (pContext->pLayerManagerContext)
        {
            if (pContext->pLayerManagerContext->readyToUse)
            {
                mspin_log_printLn(eMspinVerbosityWarn,
                        "%s(context=%p){%d} WARNING: LayerManager isn't deinitialized. Do it now!",
                        __FUNCTION__, *ppInstanceHandle, pContext->instanceId);

                //Inform higher layer not to use LayerManager anymore
                if (pContext->OnLMDeinitializedCB)
                {
                    pContext->OnLMDeinitializedCB(pContext->outsideContext);
                }

                mspin_core_deleteLMContext(pContext);
            }
            else
            {
                if (!pContext->pLayerManagerContext->pWLCtx)
                {
                    mspin_log_printLn(eMspinVerbosityDebug,
                            "%s(context=%p){%d} LM context is not in use or already freed",
                            __FUNCTION__, pContext, pContext->instanceId);
                }
                else
                {
                    int counter = 50; //at maximum wait 5 seconds
                    mspin_log_printLn(eMspinVerbosityWarn,
                            "%s(context=%p){%d} WARNING: WLCtx=%p is still valid -> wait for deletion",
                            __FUNCTION__, pContext, pContext->instanceId, pContext->pLayerManagerContext->pWLCtx);

                    while(pContext->pLayerManagerContext->pWLCtx && counter)
                    {
                        usleep(100*1000); //wait 100ms and try again
                        --counter;
                    }
                }
            }

            //Free LayerManager context only if Wayland context has been released. Otherwise skip it and cause
            // a memory leak
            if (!pContext->pLayerManagerContext->pWLCtx)
            {
                mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p){%d} Freeing LayerManager context=%p",
                        __FUNCTION__, pContext, pContext->instanceId, pContext->pLayerManagerContext);

                int rc = pthread_rwlock_destroy(&(pContext->pLayerManagerContext->rwlock));
                if (rc)
                {
                    mspin_log_printLn(eMspinVerbosityFatal, "%s() FATAL ERROR: Failed to destroy rwlock with rc=%d",
                            __FUNCTION__, rc);
                }

                rc = pthread_mutex_lock(&(pContext->LMCtxMutex));
                if (rc != 0)
                {
                    mspin_log_printLn(eMspinVerbosityFatal, "%s() FATAL ERROR: Failed to lock LMCtxMutex with rc=%d",
                            __FUNCTION__, rc);
                }
                free(pContext->pLayerManagerContext);
                pContext->pLayerManagerContext = NULL;
                rc = pthread_mutex_unlock(&(pContext->LMCtxMutex));
                if (rc != 0)
                {
                    mspin_log_printLn(eMspinVerbosityFatal, "%s() FATAL ERROR: Failed to unlock LMCtxMutex with rc=%d",
                            __FUNCTION__, rc);
                }

                mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p){%d} LayerManager context freed",
                        __FUNCTION__, pContext, pContext->instanceId);
            }
            else
            {
                //This case should not be reached. If reached, some other error occurred. Result is a memory leak here
                mspin_log_printLn(eMspinVerbosityFatal,
                        "%s(context=%p){%d} FATAL ERROR: WLCtx=%p is still present => can't free free LMCtx=%p",
                        __FUNCTION__, pContext, pContext->instanceId, pContext->pLayerManagerContext->pWLCtx,
                        pContext->pLayerManagerContext);
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(context=%p){%d} No LayerManager context to delete",
                    __FUNCTION__, pContext, pContext->instanceId);
        }

        if (pCore)
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(context=%p){%d} sleeping 200 msec to let Core=%p calm down",
                    __FUNCTION__, pContext, pContext->instanceId, pCore);
            usleep(200*1000);   // 200ms
            mspin_core_deleteCoreInstance(pContext, pCore);
            //pContext->pCoreInstance is already set to NULL in mspin_core_stopCoreInstance
        }

        mspin_conn_releaseHandle(&(pContext->pConnectionHandle));

        mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p){%d} free(pContext)",
                __FUNCTION__, *ppInstanceHandle, pContext->instanceId);
        pthread_mutex_destroy(&(pContext->LMCtxMutex));
        free(pContext);
        *ppInstanceHandle = NULL;
    }
}

void MSPIN_MeasurePerformanceAOAP(void)
{
    mspin_log_printLn(eMspinVerbosityWarn,
            "%s() WARNING: Enable performance measurements for AOAP",
            __FUNCTION__);
    mspin_conn_enableAOAPPerfomanceMeasurement();
}


void MSPIN_MeasureFrameLatency(void)
{
#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
    mspin_log_printLn(eMspinVerbosityWarn,
            "%s() WARNING: Enable frame latency measurements",
            __FUNCTION__);

    mspin_measure_enableFrameLatency();
#else
    mspin_log_printLn(eMspinVerbosityWarn,
            "%s() WARNING: Enabling frame latency measurements isn't supported.",
            __FUNCTION__);

    mspin_log_printLn(eMspinVerbosityWarn,
            "%s() WARNING: To enable check build flag 'MSPIN_ENABLE_FRAME_MEASUREMENTS' and recompile mySPIN",
            __FUNCTION__);
#endif //#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
}

void MSPIN_CountFramesPerSecond(void)
{
    mspin_core_countFPS();
}


MSPIN_ERROR MSPIN_ConnectAoap(MSPIN_Instance_t* instanceHandle, U32 vendorId, U32 productId, const char* serial, bool* pAudioSupport)
{
    MSPIN_ERROR ret = MSPIN_ERROR_GENERAL;
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    
    if (pContext && pAudioSupport)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(pContext=%p %.4x:%.4x, s/n='%s', audio=%s) entered",
                __FUNCTION__, pContext, vendorId, productId, serial ? serial : "n/a", *pAudioSupport ? "enabled" : "disabled");

        if (NULL != mspin_conn_connectAOAP(pContext, vendorId, productId, serial, pAudioSupport))
        {
            ret = mspin_core_start(pContext);
            // in case of failure the core will be deleted with MSPIN_DeleteInstance()
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                "%s(pContext=%p) ERROR: mspin_conn_connectAOAP returned NULL!",
                __FUNCTION__, pContext);
            ret = MSPIN_ERROR_CONNECTION_START;
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(pContext=%p, pAudioSupport=%p) FATAL ERROR: pContext or pAudioSupport is NULL",
             __FUNCTION__, pContext, pAudioSupport);
        return MSPIN_ERROR_GENERAL;
    }

    mspin_log_printLn(eMspinVerbosityInfo, "%s(pContext=%p) returns '%s'(%d)",
            __FUNCTION__, pContext, MSPIN_GetErrorName(ret), ret);

    return ret;
}

void MSPIN_DisconnectAoap(MSPIN_Instance_t *pInstanceHandle)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(%p)", __FUNCTION__, pInstanceHandle);

    mspin_context_t* pContext = (mspin_context_t*)pInstanceHandle;
    mspin_conn_disconnectAoap(pContext);
}

MSPIN_ERROR MSPIN_ConnectiAP2(MSPIN_Instance_t* instanceHandle, U32 vendorId, U32 productId,
        const char *pSerial, const char *pWriteDevice, const char *pReadDevice, bool hostMode)
{
    MSPIN_ERROR ret = MSPIN_ERROR_GENERAL;
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    
    if (pContext)
    {
        mspin_connectionHandle_t *pConnectionHandle = mspin_conn_connectiAP2(pContext, vendorId,
                productId, pSerial, pWriteDevice, pReadDevice, hostMode);
        if (pConnectionHandle)
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(%.4x:%.4x, s/n='%s', writeFD='%s', readFD='%s', %sMode){%d}",
                    __FUNCTION__, vendorId, productId, pSerial, pWriteDevice, pReadDevice,
                    hostMode ? "host" : "device", pContext->instanceId);

            ret = MSPIN_SUCCESS;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(%.4x:%.4x, s/n='%s', writeFD='%s', readFD='%s' ,%sMode){%d} ERROR: Could not establish connection",
                    __FUNCTION__, vendorId, productId, pSerial, pWriteDevice, pReadDevice,
                    hostMode ? "host" : "device", pContext->instanceId);
            ret = MSPIN_ERROR_CONNECTION_START;
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(%.4x:%.4x, s/n='%s', writeFD='%s', readFD='%s' ,%sMode){} ERROR: No context set",
                __FUNCTION__, vendorId, productId, pSerial, pWriteDevice, pReadDevice,
                hostMode ? "host" : "device");
        ret = MSPIN_ERROR_GENERAL;
    }

    return ret;
}

void MSPIN_DisconnectiAP2(MSPIN_Instance_t *pInstanceHandle)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(%p)", __FUNCTION__, pInstanceHandle);

    mspin_conn_disconnectiAP2();
}

MSPIN_ERROR MSPIN_SetEAPReadWriteCallbacks(MSPIN_Instance_t *pInstanceHandle, MSPIN_EAPSend eapSend_CB, MSPIN_EAPRead eapRead_CB, void *pEapContext)
{
    MSPIN_ERROR result = MSPIN_SUCCESS;
    mspin_context_t* pContext = (mspin_context_t*)pInstanceHandle;
    if (pContext && pContext->pConnectionHandle)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s()", __FUNCTION__);

        pContext->pConnectionHandle->eapSend_CB = eapSend_CB;
        pContext->pConnectionHandle->eapReceive_CB = eapRead_CB;
        pContext->pConnectionHandle->eapContext = pEapContext;
    }
    else if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(ctx=%p){%d} FATAL ERROR: instance handle is NULL",
                __FUNCTION__, pInstanceHandle, 0);
        result = MSPIN_ERROR_GENERAL;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(ctx=%p){%d} FATAL ERROR: connection handle is NULL",
                __FUNCTION__, pInstanceHandle, pContext->instanceId);
        result = MSPIN_ERROR_GENERAL;
    }
    return result;
}


MSPIN_ERROR MSPIN_EAPSessionStart(MSPIN_Instance_t *pInstanceHandle)
{
    MSPIN_ERROR result = MSPIN_SUCCESS;
    mspin_context_t* pContext = (mspin_context_t*)pInstanceHandle;

    //Start core
    if (pContext && pContext->pConnectionHandle)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(ctx=%p){%d} called",
                __FUNCTION__, pInstanceHandle, pContext->instanceId);

        pContext->EAPSessionStarted = TRUE;
        pContext->pConnectionHandle->connectionError = FALSE; //reset connection error

        //Transport connection setup is only required for host mode
        if (MSPIN_CONNECTION_IAP_HOST == pContext->pConnectionHandle->connectionType)
        {
            result = mspin_conn_iap2_setupNativeTransportConnection(pContext->pConnectionHandle->pReadDevice,
                    pContext->pConnectionHandle->pWriteDevice);
        }

        if (MSPIN_SUCCESS == result)
        {
            result = mspin_core_start(pContext);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityFatal,
                    "%s(ctx=%p){%d} ERROR: Could not setup connection",
                    __FUNCTION__, pInstanceHandle, pContext->instanceId);
        }
    }
    else if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(ctx=%p){} FATAL ERROR: pContext is NULL",
                __FUNCTION__, pInstanceHandle);
        result = MSPIN_ERROR_GENERAL;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(ctx=%p){%d} FATAL ERROR: connection handle is NULL",
                __FUNCTION__, pInstanceHandle, pContext->instanceId);
        result = MSPIN_ERROR_GENERAL;
    }

    return result;
}

void MSPIN_EAPSessionStop(MSPIN_Instance_t *pInstanceHandle)
{
    mspin_context_t* pContext = (mspin_context_t*)pInstanceHandle;

    if (pContext)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(%p){%d} entered",
                __FUNCTION__, pInstanceHandle, pContext->instanceId);

        //Reset session values to allow restart of session
        pContext->EAPSessionStarted = FALSE;
        pContext->isReady = FALSE;
        pContext->isCanceled = FALSE;
        pContext->cancelReason = MSPIN_ERROR_UNKNOWN;
        pContext->hasEverBeenReady = FALSE;
        pContext->frameTransmissionActive = FALSE;

        void* pCore = mspin_core_stopCoreInstance(pContext);
        if (pCore)
        {
            mspin_core_deleteCoreInstance(pContext, pCore);
            //'mspin_core_deleteCoreInstance' sets pContext->pCoreInstance to NULL
            mspin_log_printLn(eMspinVerbosityDebug, "%s(%p){%d} core instance deleted -> shutdown iAP2 connection",
                    __FUNCTION__, pInstanceHandle, pContext->instanceId);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(%p){%d} no core instance found -> nothing to delete",
                    __FUNCTION__, pInstanceHandle, pContext->instanceId);
        }

        mspin_conn_iap2_shutdownConnection();

        //In normal case the LayerManager should be deleted in mspin_core_onMainThreadEnd. But in case this callback
        // was not called, check if LayerManagerContext is still present and delete it
        if (pContext->pLayerManagerContext && pContext->pLayerManagerContext->readyToUse)
        {
             mspin_log_printLn(eMspinVerbosityDebug, "%s(%p){%d} Delete LayerManager context now",
                     __FUNCTION__, pInstanceHandle, pContext->instanceId);

             //Inform higher layer not to use LayerManager anymore
             if (pContext->OnLMDeinitializedCB)
             {
                 pContext->OnLMDeinitializedCB(pContext->outsideContext);
             }

             mspin_core_deleteLMContext(pContext);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(%p){} FATAL ERROR: handle is NULL",
                __FUNCTION__, pInstanceHandle);
    }
}

MSPIN_ERROR MSPIN_ConnectTCPServer(MSPIN_Instance_t *pInstanceHandle, const U8 *hostname, unsigned int port)
{
    MSPIN_ERROR ret = MSPIN_ERROR_NOT_IMPLEMENTED;
    mspin_context_t* pContext = (mspin_context_t*)pInstanceHandle;

    if (pContext)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(%p, ip='%s', port=%d){%d} entered",
            __FUNCTION__, pContext, hostname, port, pContext->instanceId);

        ret = mspin_conn_connectTCPServer(pContext, hostname, port);
        if (MSPIN_SUCCESS == ret)
        {
            ret = mspin_core_start(pContext);
            // in case of failure the core will be deleted with MSPIN_DeleteInstance()

            mspin_log_printLn(eMspinVerbosityInfo, "%s(%p, ip='%s', port=%d){%d} successful",
                __FUNCTION__, pContext, hostname, port, pContext->instanceId);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                "%s(%p, ip='%s', port=%d){%d} ERROR: mspin_conn_connectTCP failed with %d!",
                __FUNCTION__, pContext, hostname, port, pContext->instanceId, ret);
            //return result of mspin_conn_connectTCP
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(%p, ip='%s', port=%d){} ERROR: No context set",
                __FUNCTION__, pInstanceHandle, hostname, port);
        ret = MSPIN_ERROR_GENERAL;
    }

    return ret;
}

MSPIN_ERROR MSPIN_DisconnectTCPServer(MSPIN_Instance_t *pInstanceHandle)
{
    MSPIN_ERROR ret = MSPIN_ERROR_NOT_IMPLEMENTED;
    mspin_context_t* pContext = (mspin_context_t*)pInstanceHandle;

    if (pContext)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(%p){%d} entered",
            __FUNCTION__, pContext, pContext->instanceId);

        ret = mspin_conn_disconnectTCPServer(pContext);

        mspin_log_printLn(eMspinVerbosityInfo, "%s(%p){%d} returns with %d",
            __FUNCTION__, pContext, pContext->instanceId, ret);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(%p){} ERROR: No context set",
                __FUNCTION__, pInstanceHandle);
        ret = MSPIN_ERROR_GENERAL;
    }

    return ret;
}

MSPIN_ERROR MSPIN_StartUDPListener(int port, MSPIN_UDPMessageReceived mesgReceivedCB, void *pUDPListenerContext)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(udpCtx=%p) entered",
            __FUNCTION__, pUDPListenerContext);

    return mspin_conn_startUDPListener(port, mesgReceivedCB, pUDPListenerContext);
}

MSPIN_ERROR MSPIN_StopUDPListener(void)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s() entered", __FUNCTION__);

    return mspin_conn_stopUDPListener();
}

MSPIN_ERROR MSPIN_StartTCPListener(S32 port, MSPIN_OnAcceptIncomingConnection acceptCB,
        MSPIN_OnConnectionClosed closedCB, void* callbackContext)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s(port=%d, acceptCB=%p, closedCB=%p, context=%p) entered",
            __FUNCTION__, port, acceptCB, closedCB, callbackContext);

    return mspin_conn_startTCPListener(port, acceptCB, closedCB, callbackContext);
}

MSPIN_ERROR MSPIN_StopTCPListener(void)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s() entered", __FUNCTION__);

    return mspin_conn_stopTCPListener();
}

MSPIN_ERROR MSPIN_StartTLSListener(S32 port, MSPIN_TLS_CONFIGURATION_t* tlsConfig,
        MSPIN_OnAcceptIncomingConnection acceptCB, MSPIN_OnConnectionClosed closedCB, void* callbackContext)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s(port=%d, tlsConfig=%p, acceptCB=%p, closedCB=%p, context=%p) entered with config",
            __FUNCTION__, port, tlsConfig, acceptCB, closedCB, callbackContext);

    return mspin_conn_startTLSListener(port, tlsConfig, acceptCB, closedCB, callbackContext);
}

MSPIN_ERROR MSPIN_StopTLSListener(void)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s() entered", __FUNCTION__);

    return mspin_conn_stopTLSListener();
}

MSPIN_ERROR MSPIN_CloseConnection(S32 connectionID)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(id=%d) entered", __FUNCTION__, connectionID);

    return mspin_conn_closeTCPConnection(connectionID);
}

MSPIN_ERROR MSPIN_GetMacAddress(S32 connectionID, U8* macAddress)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(id=%d, macAddress=%p) entered",
            __FUNCTION__, connectionID, macAddress);
    return mspin_conn_getMACAddress(connectionID, macAddress);
}

MSPIN_ERROR MSPIN_StartUDPBroadcasting(S32 udpPort, const U8* interface, MSPIN_UDP_MESSAGE_PARAMETER_t messageParameters, U32 timeout,
        MSPIN_OnUDPBroadcastEnd onUDPBroadcastEnd, void* onUDPBroadcastEndContext)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(udpPort=%d, interface='%s', timeout=%dms, cb=%p, context=%p) entered",
            __FUNCTION__, udpPort, interface, timeout, onUDPBroadcastEnd, onUDPBroadcastEndContext);

    return mspin_conn_startUDPBroadcasting(udpPort, interface, messageParameters, timeout, onUDPBroadcastEnd, onUDPBroadcastEndContext);
}

MSPIN_ERROR MSPIN_StopUDPBroadcasting()
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s() entered", __FUNCTION__);

    return mspin_conn_stopUDPBroadcasting();
}

MSPIN_ERROR MSPIN_StartUDPUnicasting(S32 udpPort, const U8* interface, const U8* destIPAddr, MSPIN_UDP_MESSAGE_PARAMETER_t messageParameters, U32 timeout,
        MSPIN_OnUDPUnicastEnd onUDPUnicastEnd, void* onUDPUnicastEndContext)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(udpPort=%d, interface='%s', destIPAddr='%s', timeout=%dms, cb=%p, context=%p) entered",
            __FUNCTION__, udpPort, interface, (const char*)destIPAddr, timeout, onUDPUnicastEnd, onUDPUnicastEndContext);

    return mspin_conn_startUDPUnicasting(udpPort, interface, destIPAddr, messageParameters, timeout, onUDPUnicastEnd, onUDPUnicastEndContext);
}

MSPIN_ERROR MSPIN_StopUDPUnicasting(void)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s() entered", __FUNCTION__);

    return mspin_conn_stopUDPUnicasting();
}

MSPIN_ERROR MSPIN_ConnectTCPClient(MSPIN_Instance_t *pInstanceHandle, S32 connectionID)
{
    MSPIN_ERROR ret = MSPIN_ERROR_NOT_IMPLEMENTED;
    mspin_context_t* pContext = (mspin_context_t*)pInstanceHandle;

    if (pContext)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(%p, id=%d){%d} entered",
            __FUNCTION__, pContext, connectionID, pContext->instanceId);

        ret = mspin_conn_connectTCPClient(pContext, connectionID);
        if (MSPIN_SUCCESS == ret)
        {
            ret = mspin_core_start(pContext);
            // in case of failure the core will be deleted with MSPIN_DeleteInstance()

            mspin_log_printLn(eMspinVerbosityInfo, "%s(%p, id=%d){%d} successful",
                __FUNCTION__, pContext, connectionID, pContext->instanceId);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                "%s(%p, id=%d){%d} ERROR: mspin_conn_connectTCP failed with %d!",
                __FUNCTION__, pContext, connectionID, pContext->instanceId);
            //return result of mspin_conn_connectTCP
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(%p, id=%d){} ERROR: No context set",
                __FUNCTION__, pInstanceHandle, connectionID);
        ret = MSPIN_ERROR_NULL_HANDLE;
    }

    return ret;
}

MSPIN_ERROR MSPIN_DisconnectTCPClient(MSPIN_Instance_t *pInstanceHandle)
{
    MSPIN_ERROR ret = MSPIN_ERROR_NOT_IMPLEMENTED;
    mspin_context_t* pContext = (mspin_context_t*)pInstanceHandle;

    if (pContext)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(%p){%d} entered", __FUNCTION__, pContext, pContext->instanceId);

        ret = mspin_conn_disconnectTCPClient(pContext);

        mspin_log_printLn(eMspinVerbosityInfo, "%s(%p){%d} returns with %d",
            __FUNCTION__, pContext, pContext->instanceId, ret);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(%p){} ERROR: No context set",
                __FUNCTION__, pInstanceHandle);
        ret = MSPIN_ERROR_NULL_HANDLE;
    }

    return ret;
}

MSPIN_ERROR MSPIN_ConnectTLSClient(MSPIN_Instance_t *pInstanceHandle, S32 connectionID)
{
    MSPIN_ERROR ret = MSPIN_ERROR_NOT_IMPLEMENTED;
    mspin_context_t* pContext = (mspin_context_t*)pInstanceHandle;

    if (pContext)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(%p, id=%d){%d} entered",
            __FUNCTION__, pContext, connectionID, pContext->instanceId);

        ret = mspin_conn_connectTLSClient(pContext, connectionID);
        if (MSPIN_SUCCESS == ret)
        {
            ret = mspin_core_start(pContext);
            // in case of failure the core will be deleted with MSPIN_DeleteInstance()

            mspin_log_printLn(eMspinVerbosityInfo, "%s(%p, id=%d){%d} successful",
                __FUNCTION__, pContext, connectionID, pContext->instanceId);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                "%s(%p, id=%d){%d} ERROR: mspin_conn_connectTCP failed with %d!",
                __FUNCTION__, pContext, connectionID, pContext->instanceId);
            //return result of mspin_conn_connectTCP
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(%p, id=%d){} ERROR: No context set",
                __FUNCTION__, pInstanceHandle, connectionID);
        ret = MSPIN_ERROR_NULL_HANDLE;
    }

    return ret;
}

MSPIN_ERROR MSPIN_DisconnectTLSClient(MSPIN_Instance_t *pInstanceHandle)
{
    MSPIN_ERROR ret = MSPIN_ERROR_NOT_IMPLEMENTED;
    mspin_context_t* pContext = (mspin_context_t*)pInstanceHandle;

    if (pContext)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(%p){%d} entered", __FUNCTION__, pContext, pContext->instanceId);

        ret = mspin_conn_disconnectTLSClient(pContext);

        mspin_log_printLn(eMspinVerbosityInfo, "%s(%p){%d} returns with '%s'",
            __FUNCTION__, pContext, pContext->instanceId, MSPIN_GetErrorName(ret));
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(%p){} ERROR: No context set",
                __FUNCTION__, pInstanceHandle);
        ret = MSPIN_ERROR_NULL_HANDLE;
    }

    return ret;
}

void MSPIN_SetFrameProperties(MSPIN_Instance_t* instanceHandle, MSPIN_FrameProperties_t *frameProps)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext && pContext->pLayerManagerContext && frameProps)
    {
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(ctx=%p, framePros=%p) w=%d, h=%d, phyW=%d, phyH=%d, format=%d, compression=%d, prefCompression=%d, layerId=%d, surfaceId=%d",
                __FUNCTION__, instanceHandle, frameProps, frameProps->width, frameProps->height,
                frameProps->physicalWidth, frameProps->physicalHeight, (int)frameProps->format,
                (int)frameProps->compression, frameProps->preferredCompression,
                frameProps->layerId, frameProps->surfaceId);

        pContext->useRawOutput = eFLAG_FALSE;   // might be changed by setting MSPIN_OnFrameUpdateRaw

        pContext->pLayerManagerContext->pixelFormat = frameProps->format;

        if (MSPIN_PIXEL_FORMAT_RGB565 == frameProps->format)
        {
            pContext->pLayerManagerContext->bytesPerPixel = 2;
        }
        else if (MSPIN_PIXEL_FORMAT_RGB888 == frameProps->format)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(ctx=%p, framePros=%p) ERROR: Not supported pixel format set. Fall back to RGB565",
                    __FUNCTION__, instanceHandle, frameProps);

            pContext->pLayerManagerContext->pixelFormat = MSPIN_PIXEL_FORMAT_RGB565;
            pContext->pLayerManagerContext->bytesPerPixel = 2;
        }
        else if (MSPIN_PIXEL_FORMAT_ARGB8888 == frameProps->format)
        {
            pContext->pLayerManagerContext->bytesPerPixel = 4;
        }
        else if (MSPIN_PIXEL_FORMAT_RGBA8888 == frameProps->format)
        {
            pContext->pLayerManagerContext->bytesPerPixel = 4;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(ctx=%p, framePros=%p) ERROR: Not supported pixel format set. Fall back to RGB565",
                    __FUNCTION__, instanceHandle, frameProps);
            pContext->pLayerManagerContext->pixelFormat = MSPIN_PIXEL_FORMAT_RGB565;
            pContext->pLayerManagerContext->bytesPerPixel = 2;
        }

        pContext->pixelEndianess = ePIXELENDIANESS_LittleEndian;

        if ((MSPIN_FRAME_COMPRESSION_NONE == frameProps->compression)
                || (MSPIN_FRAME_COMPRESSION_UNCOMPRESSED == frameProps->compression))
        {
#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL)
            pContext->compressionFormat = eFRAMECOMPRESSION_Uncompressed;
#else
            pContext->compressionFormat = eFRAMECOMPRESSION_NONE; //eFRAMECOMPRESSION_NONE is supported since IVI Core 1.1
#endif //#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL)
        }
        else if (MSPIN_FRAME_COMPRESSION_ZLIB == frameProps->compression)
        {
            pContext->compressionFormat = eFRAMECOMPRESSION_ZLIB;
        }
        else if (MSPIN_FRAME_COMPRESSION_JPEG == frameProps->compression)
        {
            pContext->compressionFormat = eFRAMECOMPRESSION_JPEG;
#if (MSPIN_IVI_CORE_VERSION < 0x01010000UL)
            pContext->useRawOutput = eFLAG_TRUE; //use raw output for IVI Core versions before 1.1
#endif //(MSPIN_IVI_CORE_VERSION < 0x01010000UL)
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(ctx=%p, framePros=%p) ERROR: Not supported frame compression set. Fall back to ZLib compression",
                    __FUNCTION__, instanceHandle, frameProps);
            pContext->compressionFormat = eFRAMECOMPRESSION_ZLIB;
        }

        pContext->pingTimeout = MSPIN_IVI_CORE_PING_TIMEOUT;
        pContext->pLayerManagerContext->frameWidth = frameProps->width;
        pContext->pLayerManagerContext->frameHeight = frameProps->height;

#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //Supported since IVI Core 1.1
        pContext->physicalWidth = frameProps->physicalWidth;
        pContext->physicalHeight = frameProps->physicalHeight;
        pContext->preferredCompression = frameProps->preferredCompression;
#endif // (MSPIN_IVI_CORE_VERSION >= 0x01010000UL)

        pContext->pLayerManagerContext->layerId = frameProps->layerId;
        pContext->pLayerManagerContext->surfaceId = frameProps->surfaceId;

        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(ctx=%p, framePros=%p) Frame props are: w=%d, h=%d, layerId=%d, surfaceId=%d",
                __FUNCTION__, instanceHandle, frameProps, pContext->pLayerManagerContext->frameWidth,
                pContext->pLayerManagerContext->frameHeight, pContext->pLayerManagerContext->layerId,
                pContext->pLayerManagerContext->surfaceId);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p, framePros=%p) ERROR: parameters aren't valid",
                __FUNCTION__, instanceHandle, frameProps);
    }
}

void MSPIN_SetAccessoryPropertiesAoap(MSPIN_Instance_t* instanceHandle, MSPIN_AccessoryPropertiesAoap_t *accProps)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext && accProps)
    {
        strncpy(pContext->AccVendor, accProps->vendor, sizeof(pContext->AccVendor));
        pContext->AccVendor[sizeof(pContext->AccVendor)-1] = 0;
        strncpy(pContext->AccModel, accProps->model, sizeof(pContext->AccModel));
        pContext->AccModel[sizeof(pContext->AccModel)-1] = 0;
        strncpy(pContext->AccDescription, accProps->description, sizeof(pContext->AccDescription));
        pContext->AccDescription[sizeof(pContext->AccDescription)-1] = 0;
        strncpy(pContext->AccVersion, accProps->version, sizeof(pContext->AccVersion));
        pContext->AccVersion[sizeof(pContext->AccVersion)-1] = 0;
        strncpy(pContext->AccUri, accProps->uri, sizeof(pContext->AccUri));
        pContext->AccUri[sizeof(pContext->AccUri)-1] = 0;
        strncpy(pContext->AccSerial, accProps->serial, sizeof(pContext->AccSerial));
        pContext->AccSerial[sizeof(pContext->AccSerial)-1] = 0;

        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(ctx=%p, accProps=%p) AOAP accessory properties are: Vendor='%s', model='%s', description='%s'",
                __FUNCTION__, instanceHandle, accProps, pContext->AccVendor, pContext->AccModel, pContext->AccDescription);
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(ctx=%p, accProps=%p) properties continued: Version='%s', uri='%s', serial='%s'",
                __FUNCTION__, instanceHandle, accProps, pContext->AccVersion, pContext->AccUri, pContext->AccSerial);
    }
    else if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p, accProps=%p) ERROR: Context is NULL",
                __FUNCTION__, instanceHandle, accProps);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p, accProps=%p) ERROR: Accessory properties are NULL",
                __FUNCTION__, instanceHandle, accProps);
    }
}

void MSPIN_SetISO639LanguageCode(MSPIN_Instance_t* instanceHandle, U8 languageCode[3])
{
    void* pInst = NULL;
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pInst = pContext->pCoreInstance;
    }
    // calling with pInst == NULL is acceptable here:
    mySpin_SetISO639LanguageCode(pInst, languageCode);
    mspin_log_printLn(eMspinVerbosityDebug, "%s(%p) set language to %s", __FUNCTION__, instanceHandle, languageCode);
}

void MSPIN_SetCapabilities(MSPIN_Instance_t* instanceHandle, U32 caps)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->capabilities = caps;
    }
}

void MSPIN_SetPhoneCallStatusCallback(MSPIN_Instance_t* instanceHandle,
        void(*phoneCallStatusCB)(void* context, bool isActive))
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->OnPhoneCallStatus = phoneCallStatusCB;
    }
}

void MSPIN_SetAppTransitionStatusCallback(MSPIN_Instance_t* instanceHandle,
        void(*appTransitionStatusCB)(void* context,
        bool isActive))
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->OnAppTransitionStatus = appTransitionStatusCB;
    }
}

void MSPIN_SetAppInactiveCallback(MSPIN_Instance_t* instanceHandle, void(*appInactiveCB)(void* context,
        bool isInactive))
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->OnAppInactive = appInactiveCB;
    }
}


void MSPIN_SetBlockStatusCallback(MSPIN_Instance_t* instanceHandle,
        void(*blockStatusCB)(void* context, bool isBlocked))
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->OnBlockStatus = blockStatusCB;
    }
}

void MSPIN_SetPTTAvailableCallback(MSPIN_Instance_t* instanceHandle,
        void(*pttAvailableCB)(void* context, bool pttAvailable))
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->OnPTTAvailable = pttAvailableCB;
    }
}

MSPIN_ERROR MSPIN_SetLauncherStateCallback(MSPIN_Instance_t* instanceHandle,
        MSPIN_OnLauncherStateChange launcherStateChangeCB)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01020000UL) //Supported since IVI Core 1.2
        pContext->OnLauncherStateChangeCB = launcherStateChangeCB;
        return MSPIN_SUCCESS;
#else
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p, cb=%p) ERROR: Not supported before Core 1.2",
                __FUNCTION__, instanceHandle, launcherStateChangeCB);
        return MSPIN_ERROR_NOT_IMPLEMENTED;
#endif
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p, cb=%p) ERROR: Context is NULL",
                __FUNCTION__, instanceHandle, launcherStateChangeCB);
        return MSPIN_ERROR_NULL_HANDLE;
    }
}

MSPIN_ERROR MSPIN_SetHomeButtonUsableCallback(MSPIN_Instance_t* instanceHandle,
        MSPIN_OnHomeButtonUsabilityChange homeButtonUsableChangeCB)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01020000UL) //Supported since IVI Core 1.2
        pContext->OnHomeButtonUsabilityChangeCB = homeButtonUsableChangeCB;
        return MSPIN_SUCCESS;
#else
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p, cb=%p) ERROR: Not supported before Core 1.2",
                __FUNCTION__, instanceHandle, homeButtonUsableChangeCB);
        return MSPIN_ERROR_NOT_IMPLEMENTED;
#endif
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p, cb=%p) ERROR: Context is NULL",
                __FUNCTION__, instanceHandle, homeButtonUsableChangeCB);
        return MSPIN_ERROR_NULL_HANDLE;
    }
}

void MSPIN_SetInitiatePhoneCallCallback(MSPIN_Instance_t* instanceHandle,
        void(*initiatePhoneCallCB)(void* context, char* numberString, char* displayString))
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->OnInitiatePhoneCall = initiatePhoneCallCB;
    }
}

void MSPIN_SetNavigateToCallback(MSPIN_Instance_t* instanceHandle,
        void (*navigateToCB) (void* context, char* displayString, MSPIN_NAVIGATETOTYPE type,
                double longitude, double latitude, MSPIN_LocationStrings_t* locationDesciption))
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->OnNavigateTo = navigateToCB;
    }
}

void MSPIN_SetVehicleDataRequestCallback(MSPIN_Instance_t* instanceHandle,
        void (*vehicleDataRequestCB) (void* context, bool request, U8 length, U32* keyList))
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->OnVehicleDataRequest = vehicleDataRequestCB;
    }
}

void MSPIN_SetCustomDataStringCallback(MSPIN_Instance_t* instanceHandle,
        void(*customStringCB)(void* context, MSPIN_PHONEMSGSTRTYPE type, MSPIN_STRINGENCODING encoding, char* data))
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->OnCustomString = customStringCB;
    }
}

void MSPIN_SetCustomIntCallback(MSPIN_Instance_t* instanceHandle,
        void(*customIntCB)(void* context, MSPIN_PHONEMSGINTTYPE type, U16 length, U8* data))
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //Supported since IVI Core 1.1
        pContext->OnCustomInt = customIntCB;
#else
        MSPIN_UNUSED(customIntCB);
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Feature not supported before IVI Core 1.1.x", __FUNCTION__);
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01010000UL)
    }
}

MSPIN_ERROR MSPIN_SetAppListChangedCallback(MSPIN_Instance_t* instanceHandle, MSPIN_OnAppListChanged appListChangedCB)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01020000UL) //Supported since IVI Core 1.2
        pContext->OnAppListChangedCB = appListChangedCB;
        return MSPIN_SUCCESS;
#else
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p, cb=%p) ERROR: Not supported before Core 1.2",
                __FUNCTION__, instanceHandle, appListChangedCB);
        return MSPIN_ERROR_NOT_IMPLEMENTED;
#endif
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p, cb=%p) ERROR: Context is NULL",
                __FUNCTION__, instanceHandle, appListChangedCB);
        return MSPIN_ERROR_NULL_HANDLE;
    }
}

MSPIN_ERROR MSPIN_SetAppIconResponseCallback(MSPIN_Instance_t* instanceHandle, MSPIN_OnAppIconResponse appIconResponseCB)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01020000UL) //Supported since IVI Core 1.2
        pContext->OnAppIconResponseCB = appIconResponseCB;
        return MSPIN_SUCCESS;
#else
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p, cb=%p) ERROR: Not supported before Core 1.2",
                __FUNCTION__, instanceHandle, appIconResponseCB);
        return MSPIN_ERROR_NOT_IMPLEMENTED;
#endif
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p, cb=%p) ERROR: Context is NULL",
                __FUNCTION__, instanceHandle, appIconResponseCB);
        return MSPIN_ERROR_NULL_HANDLE;
    }
}

MSPIN_ERROR MSPIN_SetAppStartedResponseCallback(MSPIN_Instance_t* instanceHandle,
        MSPIN_OnAppStartedResponse appStartedResponseCB)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01020000UL) //Supported since IVI Core 1.2
        pContext->OnAppStartedResponseCB = appStartedResponseCB;
        return MSPIN_SUCCESS;
#else
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p, cb=%p) ERROR: Not supported before Core 1.2",
                __FUNCTION__, instanceHandle, appStartedResponseCB);
        return MSPIN_ERROR_NOT_IMPLEMENTED;
#endif
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p, cb=%p) ERROR: Context is NULL",
                __FUNCTION__, instanceHandle, appStartedResponseCB);
        return MSPIN_ERROR_NULL_HANDLE;
    }
}


MSPIN_ERROR MSPIN_SetVoiceSupport(MSPIN_Instance_t* instanceHandle, MSPIN_VOICESESSION_SUPPORT supportType,
        MSPIN_VOICESESSION_CONSTRAINT constraint)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
#if (MSPIN_IVI_CORE_VERSION < 0x01010000UL) //Not supported before IVI Core 1.1
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p, support=%d, constraint=%d) ERROR: Voice session isn't supported before IVI Core 1.1.x",
                __FUNCTION__, instanceHandle, supportType, constraint);
        return MSPIN_ERROR_NOT_IMPLEMENTED;
#endif //(MSPIN_IVI_CORE_VERSION < 0x01010000UL)

        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(ctx=%p, support=%d, constraint=%d) updating from support=%d, constraint=%d",
                __FUNCTION__, instanceHandle, supportType, constraint, pContext->voiceSessionSupport,
                pContext->voiceSessionConstraint = constraint);

        pContext->voiceSessionSupport = supportType;
        pContext->voiceSessionConstraint = constraint;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p, support=%d, constraint=%d) ERROR: context is NULL",
                __FUNCTION__, instanceHandle, supportType, constraint);
        return MSPIN_ERROR_INVALID_PARAMETER;
    }

    return MSPIN_SUCCESS;
}

MSPIN_ERROR MSPIN_SetVoiceSessionRequestCallback(MSPIN_Instance_t* instanceHandle,
        void(*voiceSessionRequestCB)(void* context, MSPIN_VOICESESSION_REQUESTTYPE requestType))
{
    MSPIN_ERROR result = MSPIN_SUCCESS;
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;

#if (MSPIN_IVI_CORE_VERSION < 0x01010000UL) //Not supported before IVI Core 1.1
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Feature is not supported before IVI Core 1.1", __FUNCTION__);
    result = MSPIN_ERROR_NOT_IMPLEMENTED;
#else
    if (pContext)
    {
        pContext->OnVoiceSessionRequest = voiceSessionRequestCB;
    }
#endif //(MSPIN_IVI_CORE_VERSION < 0x01010000UL)

    return result;
}

MSPIN_ERROR MSPIN_UpdateVoiceSessionStatus(MSPIN_Instance_t* instanceHandle, MSPIN_VOICESESSION_STATUS status, MSPIN_VOICESESSION_STATUS_CONSTRAINT constraint)
{
    MSPIN_ERROR result = MSPIN_SUCCESS;

#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //Supported since IVI Core 1.1
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;

    if (pContext)
    {
        if (MSPIN_VOICESESSION_NOT_SUPPORTED != pContext->voiceSessionSupport)
        {
            VoiceSessionStatusConstraint coreConstraint = eVOICESESSIONSTATUSCONSTRAINT_NoConstraint;
            switch (constraint)
            {
                case (MSPIN_VOICESESSION_STATUS_CONSTRAINT_NONE):
                {
                    coreConstraint = eVOICESESSIONSTATUSCONSTRAINT_NoConstraint;
                    break;
                }
                case (MSPIN_VOICESESSION_STATUS_CONSTRAINT_NOHFP):
                {
                    coreConstraint = eVOICESESSIONSTATUSCONSTRAINT_NoHFPConnection;
                    break;
                }
                case (MSPIN_VOICESESSION_STATUS_CONSTRAINT_VOICEALREADYACTIVE):
                {
                    coreConstraint = eVOICESESSIONSTATUSCONSTRAINT_VoiceAlreadyActive;
                    break;
                }
                case (MSPIN_VOICESESSION_STATUS_CONSTRAINT_SCOTIMEOUT):
                {
                    coreConstraint = eVOICESESSIONSTATUSCONSTRAINT_SCOTimeout;
                    break;
                }
                default:
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(%p, status=%d, constraint=%d) ERROR: Constraint value not supported => change to none",
                            __FUNCTION__, instanceHandle, status, constraint);
                    break;
                }
            }

            mspin_log_printLn(eMspinVerbosityInfo, "%s(%p, status=%d, constraint=%d)", __FUNCTION__, instanceHandle, status, constraint);
            mySpin_VoiceSessionStatus(pContext->pCoreInstance, (VoiceSessionStatus)status, coreConstraint);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(%p, status=%d, constraint=%d) ERROR: Voice session support isn't enabled",
                    __FUNCTION__, instanceHandle, status, constraint);
            result = MSPIN_ERROR_GENERAL;
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(%p, status=%d, constraint=%d) ERROR: instance is NULL", __FUNCTION__, instanceHandle, status, constraint);
        result = MSPIN_ERROR_INVALID_PARAMETER;
    }
#else
    mspin_log_printLn(eMspinVerbosityError,
            "%s(%p, status=%d, constraint=%d) ERROR: Feature not supported before IVI Core 1.1.x",
            __FUNCTION__, instanceHandle, status, constraint);
    result = MSPIN_ERROR_NOT_SUPPORTED;
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01010000UL)

    return result;
}

MSPIN_ERROR MSPIN_SetAudioRequestCallback(MSPIN_Instance_t* instanceHandle,
        void (*audioRequestCB)(void* context, MSPIN_AUDIOCONTROL command, U32 requestID, MSPIN_AUDIOTYPE type))
{
    MSPIN_ERROR result = MSPIN_SUCCESS;
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;

#if (MSPIN_IVI_CORE_VERSION < 0x01010000UL) //Not supported before IVI Core 1.1
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Feature not supported before IVI Core 1.1.x", __FUNCTION__);
    result = MSPIN_ERROR_NOT_SUPPORTED;
#else
    if (pContext)
    {
        pContext->OnAudioRequest = audioRequestCB;
    }
#endif //(MSPIN_IVI_CORE_VERSION < 0x01010000UL)

    return result;
}

MSPIN_ERROR MSPIN_SetAudioStatus(MSPIN_Instance_t* instanceHandle, U32 requestID,
        MSPIN_AUDIOSTATUS status, MSPIN_AUDIORESULTCODE reason)
{
    MSPIN_ERROR result = MSPIN_SUCCESS;

#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //Not supported before IVI Core 1.1
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(%p, id=%d, status=%d, reason=%d)",
                __FUNCTION__, instanceHandle, requestID, status, reason);
        mySpin_SetAudioStatus(pContext->pCoreInstance, requestID, (AudioStatus)status, (AudioRequestResultCode)reason);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(%p) ERROR: instance is NULL", __FUNCTION__, instanceHandle);
        result = MSPIN_ERROR_INVALID_PARAMETER;
    }
#else
    MSPIN_UNUSED(requestID);
    MSPIN_UNUSED(status);
    MSPIN_UNUSED(reason);

    mspin_log_printLn(eMspinVerbosityError, "%s(%p) ERROR: Feature not supported before IVI Core 1.1.x",
            __FUNCTION__, instanceHandle);
    result = MSPIN_ERROR_NOT_SUPPORTED;
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01010000UL)

    return result;
}


void MSPIN_SetErrorCallback(MSPIN_Instance_t* instanceHandle, void(*errorCB)(void* context, MSPIN_ERROR error))
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->OnError = errorCB;
    }
}

void MSPIN_SetLayerManagerInitializedCallback(MSPIN_Instance_t* instanceHandle, void(*LMInitializedCB)(void* context))
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->OnLMInitializedCB = LMInitializedCB;
    }
}

void MSPIN_SetLayerManagerDeinitializedCallback(MSPIN_Instance_t* instanceHandle, void(*LMDenitializedCB)(void* context))
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->OnLMDeinitializedCB = LMDenitializedCB;
    }
}

void MSPIN_SetFirstFrameRenderedCallback(MSPIN_Instance_t* instanceHandle, void(*FirstFrameRenderedCB)(void* context))
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->OnFirstFrameRenderedCB = FirstFrameRenderedCB;
    }
}

void MSPIN_SetFrameTransmissionStatusCallback(MSPIN_Instance_t* instanceHandle, void(*FrameTransmisionStatusCB)(void* context, bool start))
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->OnFrameTransmissionStatusChangeCB = FrameTransmisionStatusCB;
    }
}

void MSPIN_SetFrameUpdateRawCallback(MSPIN_Instance_t* instanceHandle, MSPIN_OnFrameUpdateRaw frameUpdateRawCB)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->OnFrameUpdateRawCB = frameUpdateRawCB;
    }
}

void MSPIN_KeyEvent(MSPIN_Instance_t* instanceHandle, MSPIN_KEY key, bool pressed)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
	if (pContext)
	{
		Softkey sk = eSOFTKEY_Home;
		bool ok = true;
		switch (key)
		{
		case MSPIN_KEY_HOME:
			sk = eSOFTKEY_Home;
			break;
		case MSPIN_KEY_BACK:
			sk = eSOFTKEY_Back;
			break;
		case MSPIN_KEY_MENU:
			sk = eSOFTKEY_Menu;
			break;
		case MSPIN_KEY_SEARCH:
			sk = eSOFTKEY_Search;
			break;
		default:
			ok = false;
			mspin_log_printLn(eMspinVerbosityError,
			        "%s() ERROR: illegal key %d", __FUNCTION__, key);
			break;
		}
		SoftkeyEventType ev = eSOFTKEYEVENTTYPE_Press;
		if (!pressed)
		{
			ev = eSOFTKEYEVENTTYPE_Release;
		}
		if (ok && pContext->pCoreInstance)
		{
		    mspin_log_printLn(eMspinVerbosityDebug, "%s(inst=%p, key=%d, %s) sending",
		            __FUNCTION__, instanceHandle, key, pressed ? "pressed" : "released");

			mySpin_SoftkeyEvent(pContext->pCoreInstance, sk, ev);
		}
	    else
	    {
	        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: no connection", __FUNCTION__);
	    }
	}
}

MSPIN_ERROR MSPIN_SendPosition(MSPIN_Instance_t* instanceHandle, char* positionNMEA)
{
    MSPIN_ERROR result = MSPIN_SUCCESS;
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext && pContext->pCoreInstance && positionNMEA)
    {
        if (pContext->isReady)
        {
#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //IVI Core 1.1 or newer
            char buf[MAXBUFLEN];
            int cnt = snprintf(buf, MAXBUFLEN, "{\"value\":%s}", positionNMEA);
            if ((cnt>0) && (cnt<MAXBUFLEN))
            {
                MSPIN_SetVehicleData(instanceHandle, VEHICLE_DATA_KEY_GEOLOCATION_NMEA, MSPIN_STRINGENCODING_UTF8, buf);
            }
#else
            mySpin_PositionInformation(pContext->pCoreInstance, positionNMEA);
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01010000UL)
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: IVI Core not ready", __FUNCTION__);
            result = MSPIN_ERROR_NOT_READY;
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: no connection", __FUNCTION__);
        result = MSPIN_ERROR_NULL_HANDLE;
    }
    return result;
}

void MSPIN_SendCustomString(MSPIN_Instance_t* instanceHandle, MSPIN_HOSTMSGSTRTYPE type, const char* msgString)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext && pContext->pCoreInstance)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s() %d %s", __FUNCTION__, type, msgString);
#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //IVI Core 1.1 or newer
        mySpin_CustomDataString(pContext->pCoreInstance, (ClientCustomDataStringType)type, eSTRINGENCODING_ASCII_LATIN1, msgString);
#else
        mySpin_CustomMessage(pContext->pCoreInstance, (ClientCustomMessageType)type, msgString);
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01010000UL)
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: no connection", __FUNCTION__);
    }
}

MSPIN_ERROR MSPIN_SendCustomInt(MSPIN_Instance_t* instanceHandle, MSPIN_HOSTMSGINTTYPE type, U16 length, U8* data)
{
    MSPIN_ERROR result = MSPIN_SUCCESS;
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext && pContext->pCoreInstance)
    {
        if (pContext->isReady)
        {
#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //Supported since IVI Core 1.1
            mySpin_CustomDataInt(pContext->pCoreInstance, (ClientCustomDataIntType)type, length, data);
#else
            // due to different signature not implemented for Core 1.0
            MSPIN_UNUSED(type);
            MSPIN_UNUSED(length);
            MSPIN_UNUSED(data);
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not implemented", __FUNCTION__);

            result = MSPIN_ERROR_NOT_IMPLEMENTED;
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01010000UL)
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: IVI Core not ready", __FUNCTION__);
            result = MSPIN_ERROR_NOT_READY;
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: no connection", __FUNCTION__);
        result = MSPIN_ERROR_NULL_HANDLE;
    }

    return result;
}

MSPIN_ERROR MSPIN_EnableCustomKeys(MSPIN_Instance_t* instanceHandle, MSPIN_CUSTOMDATA_KEYPROFILE profile)
{
    MSPIN_ERROR returnCode = MSPIN_SUCCESS;

#if (MSPIN_IVI_CORE_VERSION >= 0x01020000UL) //Supported since IVI Core 1.2
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, profile=%d)",
                __FUNCTION__, instanceHandle, (int)profile);

        pContext->keyProfile = profile;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, profile=%d) ERROR: Context is NULL",
                __FUNCTION__, instanceHandle, (int)profile);

        returnCode = MSPIN_ERROR_NULL_HANDLE;
    }
#else
    mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, profile=%d) ERROR: Not implemented before before Core 1.2",
            __FUNCTION__, instanceHandle, (int)profile);
    returnCode = MSPIN_ERROR_NOT_IMPLEMENTED;
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01020000UL)
    return returnCode;
}

MSPIN_ERROR MSPIN_EnablePTTCustomKey(MSPIN_Instance_t* instanceHandle, bool enable)
{
    MSPIN_ERROR returnCode = MSPIN_SUCCESS;

#if (MSPIN_IVI_CORE_VERSION >= 0x01020000UL) //Supported since IVI Core 1.2
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, %s)",
                __FUNCTION__, instanceHandle, enable ? "enable" : "disable");
        pContext->customPTTKey = enable;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, %s) ERROR: Context is NULL",
                __FUNCTION__, instanceHandle, enable ? "enable" : "disable");

        returnCode = MSPIN_ERROR_NULL_HANDLE;
    }
#else
    mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, %s) ERROR: Not implemented before before Core 1.2",
            __FUNCTION__, instanceHandle, enable ? "enable" : "disable");
    returnCode = MSPIN_ERROR_NOT_IMPLEMENTED;
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01020000UL)
    return returnCode;
}

MSPIN_ERROR MSPIN_SendCustomKey(MSPIN_Instance_t* instanceHandle, MSPIN_CUSTOMDATA_KEYTYPE type, MSPIN_CUSTOMDATA_KEYCODE code)
{
    MSPIN_ERROR returnCode = MSPIN_SUCCESS;
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext && pContext->pCoreInstance)
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //Supported since IVI Core 1.1
        if (pContext->isReady)
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, type=%d, code=%d) entered",
                    __FUNCTION__, instanceHandle, type, code);
            eReturnCode coreResult = mySpin_CustomDataKey(pContext->pCoreInstance, (ClientCustomDataKeyType)type, (ClientCustomDataKeyCode)code);
            if (eRETURN_OKAY != coreResult)
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, type=%d, code=%d) ERROR: Sending failed with=%d",
                        __FUNCTION__, instanceHandle, type, code, coreResult);
                returnCode = MSPIN_ERROR_CORE;
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, type=%d, code=%d) sent successfully",
                        __FUNCTION__, instanceHandle, type, code);
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, type=%d, code=%d) ERROR: IVI Core isn't ready",
                    __FUNCTION__, instanceHandle, type, code);
            returnCode = MSPIN_ERROR_NOT_READY;
        }
#else
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, type=%d, code=%d) ERROR: Not implemented before before Core 1.1",
                __FUNCTION__, instanceHandle, type, code);
        returnCode = MSPIN_ERROR_NOT_IMPLEMENTED;
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01010000UL)
    }
    else if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, type=%d, code=%d) ERROR: Context is NULL",
                __FUNCTION__, instanceHandle, type, code);
        returnCode = MSPIN_ERROR_NULL_HANDLE;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, type=%d, code=%d) ERROR: core instance is NULL",
                __FUNCTION__, instanceHandle, type, code);
        returnCode = MSPIN_ERROR_NULL_HANDLE;
    }

    return returnCode;
}

void MSPIN_SetNightMode(MSPIN_Instance_t* instanceHandle, bool isNight)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext && pContext->pCoreInstance)
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //IVI Core 1.1 or newer
        char buf[MAXBUFLEN];
        int cnt = snprintf(buf, MAXBUFLEN, "{\"value\":%s}", isNight ? "true" : "false");
        if ((cnt>0) && (cnt<MAXBUFLEN))
        {
            MSPIN_SetVehicleData(instanceHandle, VEHICLE_DATA_KEY_IS_NIGHT, MSPIN_STRINGENCODING_UTF8, buf);
        }
#else
        Flag night = eFLAG_FALSE;
        if (isNight)
        {
            night = eFLAG_TRUE;
        }
        mySpin_SetNightMode(pContext->pCoreInstance, night);
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01010000UL)
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: no connection", __FUNCTION__);
    }
}

void MSPIN_SetMovingStatus(MSPIN_Instance_t* instanceHandle, bool isMoving)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext && pContext->pCoreInstance)
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //IVI Core 1.1 or newer
        char buf[MAXBUFLEN];
        int cnt = snprintf(buf, MAXBUFLEN, "{\"value\":%s}", isMoving ? "true" : "false");
        if ((cnt>0) && (cnt<MAXBUFLEN))
        {
            MSPIN_SetVehicleData(instanceHandle, VEHICLE_DATA_KEY_IS_MOVING, MSPIN_STRINGENCODING_UTF8, buf);
        }
#else
        Flag moving = eFLAG_FALSE;
        if (isMoving)
        {
            moving = eFLAG_TRUE;
        }
        mySpin_SetMovingStatus(pContext->pCoreInstance, moving);
#endif // (MSPIN_IVI_CORE_VERSION >= 0x01010000UL)
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: no connection", __FUNCTION__);
    }
}

MSPIN_ERROR MSPIN_SetVehicleData(MSPIN_Instance_t* instanceHandle, U32 key, MSPIN_STRINGENCODING encoding, char* jsonString)
{
    MSPIN_ERROR returnCode = MSPIN_SUCCESS;
#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //Supported since IVI Core 1.1
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;

    if (jsonString)
    {
        if (pContext && pContext->pCoreInstance)
        {
            if (pContext->isReady)
            {
                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(ctx=%p, key=%u, str='%s')",
                        __FUNCTION__, instanceHandle, key, jsonString);

                eReturnCode coreCode = mySpin_VehicleData(pContext->pCoreInstance, (UInt32)key, (StringEncoding)encoding, jsonString);
                if (eRETURN_OKAY != coreCode)
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(ctx=%p, key=%u, str='%s') ERROR: mySpin_VehicleData returned failure=%d",
                            __FUNCTION__, instanceHandle, key, jsonString, coreCode);
                    returnCode = MSPIN_ERROR_CORE;
                }
            }
            else
            {
                returnCode = MSPIN_ERROR_NOT_READY;
                mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, key=%u, str='%s') ERROR: mySPIN not ready",
                        __FUNCTION__, instanceHandle, key, jsonString);
            }
        }
        else if (!pContext)
        {
            returnCode = MSPIN_ERROR_NULL_HANDLE;
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, key=%u, str='%s') ERROR: no connection",
                    __FUNCTION__, instanceHandle, key, jsonString);
        }
        else
        {
            returnCode = MSPIN_ERROR_NULL_HANDLE;
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, key=%u, str='%s') ERROR: No Core instance",
                    __FUNCTION__, instanceHandle, key, jsonString);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, key=%u, str='%s') ERROR: jsonString is NULL",
                __FUNCTION__, instanceHandle, key, jsonString);
        returnCode = MSPIN_ERROR_NULL_HANDLE;
    }
#else
    MSPIN_UNUSED(instanceHandle);
    MSPIN_UNUSED(key);
    MSPIN_UNUSED(encoding);
    MSPIN_UNUSED(jsonString);

    mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, key=%u, str='%s') ERROR: Not supported with Core before 1.1",
            __FUNCTION__, instanceHandle, key, jsonString);
    returnCode = MSPIN_ERROR_NOT_SUPPORTED;
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01010000UL)

    return returnCode;
}

MSPIN_ERROR MSPIN_SetVehicleDataFilter(MSPIN_Instance_t* instanceHandle, BOOL callbackOn, U8 length, U32* keyList)
{
    MSPIN_ERROR returnCode = MSPIN_ERROR_NOT_IMPLEMENTED;

#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //Supported since IVI Core 1.1
#ifdef MYSPIN_IVI_CORE_ENGINEERING
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;

    if ( (keyList == NULL) && (length != 0) )
    {
        returnCode = MSPIN_ERROR_INVALID_PARAMETER;
    }
    else
    {
        if (pContext && pContext->pCoreInstance)
        {
#ifndef LSIM
            returnCode = MSPIN_SUCCESS;
            mspin_log_printLn(eMspinVerbosityInfo,
                    "%s(ctx=%p, callback=%s, len=%d, keyList=%p) ",
                    __FUNCTION__, instanceHandle, callbackOn ? "on" : "off", length, keyList);
            eReturnCode coreCode = mySpin_Eng_SetVehicleDataFilter(pContext->pCoreInstance,(Flag)callbackOn,(UInt8)length,(UInt32*)keyList);
            if (eRETURN_OKAY != coreCode)
            {

                mspin_log_printLn(eMspinVerbosityError,
                        "%s(ctx=%p, callback=%s, len=%d, keyList=%p) ERROR: mySpin_Eng_SetVehicleDataFilter returned failure=%d",
                        __FUNCTION__, instanceHandle, callbackOn ? "on" : "off", length, keyList, coreCode);
                returnCode = MSPIN_ERROR_CORE;
            }
#else //LSIM
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported for LSIM", __FUNCTION__);
            returnCode = MSPIN_ERROR_NOT_SUPPORTED;
#endif //LSIM
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: no connection", __FUNCTION__);
            returnCode = MSPIN_ERROR_NULL_HANDLE;
        }
    }
#else //MYSPIN_IVI_CORE_ENGINEERING
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Engineering not enabled", __FUNCTION__);
    returnCode = MSPIN_ERROR_NOT_SUPPORTED;
    MSPIN_UNUSED(instanceHandle);
    MSPIN_UNUSED(callbackOn);
    MSPIN_UNUSED(length);
    MSPIN_UNUSED(keyList);
#endif //MYSPIN_IVI_CORE_ENGINEERING
#else
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported before IVI Core 1.1.x", __FUNCTION__);
    returnCode = MSPIN_ERROR_NOT_SUPPORTED;
    MSPIN_UNUSED(instanceHandle);
    MSPIN_UNUSED(callbackOn);
    MSPIN_UNUSED(length);
    MSPIN_UNUSED(keyList);
#endif // (MSPIN_IVI_CORE_VERSION >= 0x01010000UL)

    return returnCode;
}

void MSPIN_PhoneCallStatus(MSPIN_Instance_t* instanceHandle, MSPIN_PHONECALLSTATUS status)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext && pContext->pCoreInstance)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(%d)", __FUNCTION__, status);
        mySpin_PhoneCallStatusResponse(pContext->pCoreInstance, (PhoneCallStatus)status);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: no connection", __FUNCTION__);
    }
}

void MSPIN_PumpEvents(MSPIN_Instance_t* instanceHandle, S32 timeout)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        mspin_lm_pumpEvents(pContext->pLayerManagerContext, timeout);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Context is NULL", __FUNCTION__);
    }
}

void MSPIN_SetVerbosity(S32 verbosityLevel, bool prependTime)
{
    mspin_log_setVerbosityLevel(verbosityLevel, prependTime);
}

void MSPIN_SetLoggingDestination(MSPIN_LOGGING_DESTINATION type, bool enable)
{
    mspin_log_setDestination(type, enable, NULL);
}

const char* MSPIN_GetErrorName(MSPIN_ERROR error)
{
    switch (error)
    {
        case MSPIN_SUCCESS:
        {
            return "SUCCESS";
            //break;
        }
        case MSPIN_ERROR_CONNECTION_START:
        {
            return "CONNECTION_ERROR";
            //break;
        }
        case MSPIN_ERROR_CORE:
        {
            return "CORE_INIT_FAILED";
            //break;
        }
        case MSPIN_ERROR_LAYERMANAGER:
        {
            return "LAYERMANAGER_ERROR";
            //break;
        }
        case MSPIN_ERROR_GENERAL:
        {
            return "GENERAL_ERROR";
            //break;
        }
        case MSPIN_ERROR_CONNECTION_TIMEOUT:
        {
            return "CONNECTION_TIMEOUT";
            //break;
        }
        case MSPIN_ERROR_CONNECTION_EOF_ZERO:
        {
            return "EOF_CONNECTION_ERROR";
            //break;
        }
        case MSPIN_ERROR_PROTOCOL_STOP:
        {
            return "PROTOCOL_STOP_ERROR";
            //break;
        }
        case MSPIN_ERROR_Z_DATA_ERROR:
        {
            return "UNCOMPRESSING_ERROR";
            //break;
        }
        case MSPIN_ERROR_MEM_ALLOC:
        {
            return "MEMORY_FAULT";
            //break;
        }
        case MSPIN_ERROR_FRAME_SIZE_MISMATCH:
        {
            return "FRAME_SIZE_MISMATCH";
            //break;
        }
        case MSPIN_ERROR_PROTOCOL_SETUP:
        {
            return "PROTOCOL_SETUP_TIMEOUT";
            //break;
        }
        case MSPIN_ERROR_RECEIVER_START_FAILED:
        {
            return "RECEIVER_START_FAILED";
            //break;
        }
        case MSPIN_ERROR_MESSAGE_PROCESSING_FAILED:
        {
            return "MESSAGE_PROCESSING_FAILED";
            //break;
        }
        case MSPIN_ERROR_NOT_IMPLEMENTED:
        {
            return "NOT_IMPLEMENTED";
            //break;
        }
        case MSPIN_ERROR_INVALID_PARAMETER:
        {
            return "INVALID PARAMETER";
            //break;
        }
        case MSPIN_ERROR_NULL_HANDLE:
        {
            return "NULL HANDLE GIVEN";
            //break;
        }
        case MSPIN_ERROR_NOT_SUPPORTED:
        {
            return "Feature not supported";
            //break;
        }
        case MSPIN_ERROR_NOT_READY:
        {
            return "Not ready";
            //break;
        }
        case MSPIN_ERROR_JPEG:
        {
            return "JPEG error";
            //break;
        }
        case MSPIN_ERROR_INITIALIZATION_ABORT:
        {
            return "Initialization abort";
            //break;
        }
        case MSPIN_ERROR_NETWORK_NOT_REACHABLE:
        {
            return "Network not reachable";
            //break;
        }
        case MSPIN_ERROR_CONNECTION_REFUSED:
        {
            return "Connection refused";
            //break;
        }
        case MSPIN_ERROR_NOT_FOUND:
        {
            return "Not found";
            //break;
        }
        case MSPIN_ERROR_ALREADY_RUNNING:
        {
            return "Already running";
            //break;
        }
        case MSPIN_ERROR_NOT_ALLOWED:
        {
            return "Not allowed";
            //break;
        }
        case MSPIN_ERROR_UNKNOWN:
        default:
        {
            return "UNKNOWN";
            //break;
        }
    }
}

void MSPIN_SetWaylandTouch(MSPIN_Instance_t* instanceHandle, bool enable)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext && pContext->pLayerManagerContext)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(handle=%p, %s Wayland touch)",
                __FUNCTION__, pContext, enable ? "enable" : "disable");
        pContext->pLayerManagerContext->useWaylandTouch = enable;
    }
    else if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(handle=%p, %s Wayland touch) ERROR: handle is NULL",
                __FUNCTION__, pContext, enable ? "enable" : "disable");
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(handle=%p, %s Wayland touch) ERROR: LM handle is NULL",
                __FUNCTION__, pContext, enable ? "enable" : "disable");
    }
}

MSPIN_ERROR MSPIN_EnableTouchFiltering(MSPIN_Instance_t* instanceHandle, bool enable)
{
    MSPIN_ERROR result = MSPIN_SUCCESS;
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext && pContext->pLayerManagerContext)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(handle=%p, %s filtering)",
                __FUNCTION__, pContext, enable ? "enable" : "disable");
        pContext->pLayerManagerContext->useTouchFiltering = enable;
    }
    else if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(handle=%p, %s filtering) ERROR: handle is NULL",
                __FUNCTION__, pContext, enable ? "enable" : "disable");
        result = MSPIN_ERROR_INVALID_PARAMETER;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(handle=%p, %s filtering) ERROR: LM context is NULL",
                __FUNCTION__, pContext, enable ? "enable" : "disable");
        result =  MSPIN_ERROR_INVALID_PARAMETER;
    }

    return result;
}

MSPIN_ERROR MSPIN_EnablePointerFiltering(MSPIN_Instance_t* instanceHandle, bool enable)
{
    MSPIN_ERROR result = MSPIN_SUCCESS;
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext && pContext->pLayerManagerContext)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(handle=%p, %s filtering)",
                __FUNCTION__, pContext, enable ? "enable" : "disable");
        pContext->pLayerManagerContext->usePointerFiltering = enable;
    }
    else if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(handle=%p, %s filtering) ERROR: handle is NULL",
                __FUNCTION__, pContext, enable ? "enable" : "disable");
        result = MSPIN_ERROR_INVALID_PARAMETER;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(handle=%p, %s filtering) ERROR: LM context is NULL",
                __FUNCTION__, pContext, enable ? "enable" : "disable");
        result = MSPIN_ERROR_INVALID_PARAMETER;
    }

    return result;
}

void MSPIN_TouchEvents(MSPIN_Instance_t* instanceHandle, MSPIN_TouchEvent_t* eventArray, U8 numberOfEvents)
{
    TouchEvent events[numberOfEvents];
    U8 i = 0;
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;

    if (pContext && pContext->pCoreInstance)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(%d)", __FUNCTION__, numberOfEvents);

        for (i = 0; i < numberOfEvents; i++)
        {
            events[i].event = (TouchType)(eventArray[i].event);
            events[i].xPosition = eventArray[i].xPosition;
            events[i].yPosition = eventArray[i].yPosition;
            events[i].fingerId = eventArray[i].fingerId;
        }

        mySpin_TouchEvents(pContext->pCoreInstance, events, numberOfEvents);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: no connection", __FUNCTION__);
    }
}

MSPIN_ERROR MSPIN_SuspendConnection(MSPIN_Instance_t* instanceHandle)
{
    if (!instanceHandle)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(handle=%p) ERROR: Instance is NULL",
                    __FUNCTION__, instanceHandle);
        return MSPIN_ERROR_NULL_HANDLE;
    }
    else
    {
        mspin_context_t* pContext = (mspin_context_t*)instanceHandle;

        if (!pContext->isSuspended)
        {
            mspin_log_printLn(eMspinVerbosityInfo,
                    "%s(handle=%p) suspend mySPIN",
                    __FUNCTION__, instanceHandle);

            //TP: iPhone crashes 2018-03: mySpin_FrameRequestStatus(pContext->pCoreInstance, eFLAG_TRUE);
            pContext->isSuspended = TRUE;

            //Issue frame transmission stop
            mspin_core_updateFrameTransmissionStatus(pContext, eFLAG_FALSE);

            return MSPIN_SUCCESS;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(handle=%p) ERROR: mySPIN connection already suspended",
                    __FUNCTION__, instanceHandle);
            return MSPIN_ERROR_GENERAL;
        }
    }
}

MSPIN_ERROR MSPIN_ResumeConnection(MSPIN_Instance_t* instanceHandle)
{
    if (!instanceHandle)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(handle=%p) ERROR: Instance is NULL",
                __FUNCTION__, instanceHandle);
        return MSPIN_ERROR_NULL_HANDLE;
    }
    else
    {
        mspin_context_t* pContext = (mspin_context_t*)instanceHandle;

        if (!pContext->isSuspended)
        {
            mspin_log_printLn(eMspinVerbosityInfo,
                    "%s(handle=%p) Resume called in active mode -> request new frame and issue first frame rendered CB",
                    __FUNCTION__, instanceHandle);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(handle=%p) resume mySPIN",
                    __FUNCTION__, instanceHandle);
        }

        //TP: iPhone crashes 2018-03: mySpin_FrameRequestStatus(pContext->pCoreInstance, eFLAG_FALSE);
        pContext->isSuspended = FALSE;

        //Request a new complete frame
        pContext->issueFirstFrameRendered = TRUE;
        if (!pContext->appTransitionInProgress)
        {
            mspin_core_requestFrameUpdate(pContext, FALSE);

            //Reset state to make sure that frameTransmissionStatusCallback is issued even if
            // there is no change (because connection got resumed without suspend)
            if (pContext->frameTransmissionActive)
            {
                pContext->frameTransmissionActive = FALSE;
                mspin_log_printLn(eMspinVerbosityInfo, "%s(handle=%p) frameTransmission is active -> mark it as inactive to issue 'FrameTransmisionStatusCB' on next frame",
                        __FUNCTION__, instanceHandle);
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(handle=%p) appTransition in progress -> send frame request in appTransition(end)",
                    __FUNCTION__, instanceHandle);
        }

        return MSPIN_SUCCESS;
    }
}

void MSPIN_SetInitializationTimeout(MSPIN_Instance_t* instanceHandle, U8 seconds)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
        pContext->initializationTimeout = seconds;
    }
}

U32 MSPIN_GetAvailableKeys(MSPIN_Instance_t* instanceHandle)
{
    U32 keys = 0;
    UInt8 numberOfKeys = 0;
    Softkey* keylist = 0;
    U8 k;

    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext && pContext->pCoreInstance)
    {
        mySpin_GetAvailableSoftkeys(pContext->pCoreInstance, &numberOfKeys, &keylist);
        if (keylist)
        {
            for (k = 0; k < numberOfKeys; ++k)
            {
                switch (keylist[k])
                {
                case eSOFTKEY_Home:
                    keys |= 1;
                    break;
                case eSOFTKEY_Back:
                    keys |= 2;
                    break;
                case eSOFTKEY_Menu:
                    keys |= 4;
                    break;
                case eSOFTKEY_Search:
                    keys |= 8;
                    break;
                default:
                    mspin_log_printLn(eMspinVerbosityWarn,
                            "%s(ctx=%p){%d} WARNING: got unknown key %d",
                            __FUNCTION__, pContext, pContext->instanceId, keylist[k]);
                    break;
                }
            }

            mspin_log_printLn(eMspinVerbosityInfo,
                    "%s(ctx=%p){%d} %d key(s) with mask %d",
                    __FUNCTION__, pContext, pContext->instanceId, numberOfKeys, keys);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityWarn, "%s(ctx=%p){%d} WARNING: keylist failure",
                    __FUNCTION__, pContext, pContext->instanceId);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p){%d} ERROR: no connection",
                __FUNCTION__, pContext, pContext ? pContext->instanceId : (U32)-1);
    }

    return keys;
}

U32 MSPIN_GetServerCapabilities(MSPIN_Instance_t* instanceHandle)
{
    U32 caps = 0;

    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext && pContext->pCoreInstance)
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL)
        caps = mySpin_GetServerCapabilities(pContext->pCoreInstance);
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(ctx=%p){%d} caps %d", __FUNCTION__, pContext,
                pContext->instanceId, caps);
#else
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p){%d} ERROR: Not supported before IVI Core 1.1.x",
                __FUNCTION__, pContext, pContext ? pContext->instanceId : (U32)-1);
#endif //#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL)
    }
    else if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p){-1} ERROR: Null context",
                __FUNCTION__, pContext);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p){%d} ERROR: no connection",
                __FUNCTION__, pContext, pContext ? pContext->instanceId : (U32)-1);
    }

    return caps;
}

MSPIN_ERROR MSPIN_GetServerOSName(MSPIN_Instance_t* instanceHandle, char* osName, U32 length)
{
#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //IVI Core 1.1 or newer
    U16 rc = 0;
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01010000UL)
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;

    if (pContext && pContext->pCoreInstance)
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //IVI Core 1.1 or newer
        rc = mySpin_GetServerOSName(pContext->pCoreInstance, osName, length);
        if (0 == rc)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, osName=%p, len=%d){%d} ERROR: Buffer too small, empty name or other error (name='%s')",
                    __FUNCTION__, pContext, osName, length, pContext ? pContext->instanceId : (U32)-1, osName ? osName : "n/a");
            return MSPIN_ERROR_GENERAL;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, osName=%p, len=%d){%d} OS name is '%s'",
                    __FUNCTION__, pContext, osName, length, pContext ? pContext->instanceId : (U32)-1, osName);
            return MSPIN_SUCCESS;
        }
#else
        MSPIN_UNUSED(length);
        MSPIN_UNUSED(osName);
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p){%d} ERROR: Not supported with IVI Core 1.0.x",
                __FUNCTION__, pContext, pContext ? pContext->instanceId : (U32)-1);
        return MSPIN_ERROR_NOT_IMPLEMENTED;
#endif // (MSPIN_IVI_CORE_VERSION >= 0x01010000UL)
    }
    else if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p){-1} ERROR: Null context",
                __FUNCTION__, pContext);
        return MSPIN_ERROR_NULL_HANDLE;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p){%d} ERROR: no connection",
                __FUNCTION__, pContext, pContext ? pContext->instanceId : (U32)-1);
        return MSPIN_ERROR_CORE;
    }
}

MSPIN_ERROR MSPIN_CreateAppInfoList(MSPIN_Instance_t* instanceHandle, U32* numberOfItems, MSPIN_AppInfoList_t** infoList)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01020000UL) //Supported since IVI Core 1.2
        AppInfoList* coreAppList = 0;
        eReturnCode rc = mySpin_CreateAppInfoList(pContext->pCoreInstance, numberOfItems, &coreAppList);
        mspin_log_printLn(eMspinVerbosityInfo, "%s(context=%p) size=%d, rc=%d",
                __FUNCTION__, instanceHandle, *numberOfItems, rc);
        if (rc == eRETURN_OKAY)
        {
            *infoList = (MSPIN_AppInfoList_t*)coreAppList;
            return MSPIN_SUCCESS;
        }
        else
        {
            return MSPIN_ERROR_CORE;
        }
#else
        MSPIN_UNUSED(numberOfItems);
        MSPIN_UNUSED(infoList);
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p) ERROR: Not supported before Core 1.2",
                __FUNCTION__, instanceHandle);
        return MSPIN_ERROR_NOT_IMPLEMENTED;
#endif
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p) ERROR: Context is NULL",
                __FUNCTION__, instanceHandle);
        return MSPIN_ERROR_NULL_HANDLE;
    }
}

MSPIN_ERROR MSPIN_DeleteAppInfoList(MSPIN_Instance_t* instanceHandle, MSPIN_AppInfoList_t* infoList)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01020000UL) //Supported since IVI Core 1.2
        eReturnCode rc = mySpin_DeleteAppInfoList(pContext->pCoreInstance, (AppInfoList*)infoList);
        mspin_log_printLn(eMspinVerbosityInfo, "%s(context=%p)",
                __FUNCTION__, instanceHandle);
        if (rc == eRETURN_OKAY)
        {
            return MSPIN_SUCCESS;
        }
        else
        {
            return MSPIN_ERROR_CORE;
        }
#else
        MSPIN_UNUSED(infoList);
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p) ERROR: Not supported before Core 1.2",
                __FUNCTION__, instanceHandle);
        return MSPIN_ERROR_NOT_IMPLEMENTED;
#endif
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p) ERROR: Context is NULL",
                __FUNCTION__, instanceHandle);
        return MSPIN_ERROR_NULL_HANDLE;
    }
}

MSPIN_ERROR MSPIN_RequestAppIcon(MSPIN_Instance_t* instanceHandle, const char* identifierURL,
        U16 width, U16 height)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01020000UL) //Supported since IVI Core 1.2
        eReturnCode rc = mySpin_RequestAppIcon(pContext->pCoreInstance, identifierURL, width, height);
        mspin_log_printLn(eMspinVerbosityInfo, "%s(context=%p) %s",
                __FUNCTION__, instanceHandle, identifierURL);
        if (rc == eRETURN_OKAY)
        {
            return MSPIN_SUCCESS;
        }
        else
        {
            return MSPIN_ERROR_CORE;
        }
#else
        MSPIN_UNUSED(identifierURL);
        MSPIN_UNUSED(width);
        MSPIN_UNUSED(height);
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p) ERROR: Not supported before Core 1.2",
                __FUNCTION__, instanceHandle);
        return MSPIN_ERROR_NOT_IMPLEMENTED;
#endif
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p) ERROR: Context is NULL",
                __FUNCTION__, instanceHandle);
        return MSPIN_ERROR_NULL_HANDLE;
    }
}

MSPIN_ERROR MSPIN_StartApp(MSPIN_Instance_t* instanceHandle, const char* identifierURL,
        MSPIN_KeyValueList_t* launchModes, MSPIN_KeyValueList_t* options)
{
    mspin_context_t* pContext = (mspin_context_t*)instanceHandle;
    if (pContext)
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01020000UL) //Supported since IVI Core 1.2
        eReturnCode rc = mySpin_StartApp(pContext->pCoreInstance, identifierURL, (KeyValueList*)launchModes, (KeyValueList*)options);
        mspin_log_printLn(eMspinVerbosityInfo, "%s(context=%p) %s",
                __FUNCTION__, instanceHandle, identifierURL);
        if (rc == eRETURN_OKAY)
        {
            return MSPIN_SUCCESS;
        }
        else
        {
            return MSPIN_ERROR_CORE;
        }
#else
        MSPIN_UNUSED(identifierURL);
        MSPIN_UNUSED(launchModes);
        MSPIN_UNUSED(options);
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p) ERROR: Not supported before Core 1.2",
                __FUNCTION__, instanceHandle);
        return MSPIN_ERROR_NOT_IMPLEMENTED;
#endif
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p) ERROR: Context is NULL",
                __FUNCTION__, instanceHandle);
        return MSPIN_ERROR_NULL_HANDLE;
    }
}

